# Core simulation for boolean which‑path marks (Present‑Act V2 ties‑only benchmark)
import json, math, hashlib
from pathlib import Path
import numpy as np, pandas as pd
from .utils import linfit

def bernoulli_draws(peff: float, n: int, rng: np.random.Generator):
    if peff <= 0.0:  return np.zeros(n, dtype=bool)
    if peff >= 1.0:  return np.ones(n, dtype=bool)
    return rng.random(n) < peff

def run_sim_grid(p_list, N_list, K_per_seed, seeds, outdir: Path):
    outdir = Path(outdir)
    outdir.mkdir(parents=True, exist_ok=True)
    # Instrument (read-only provenance; does not steer acceptance)
    windows_centers = (0.0, 0.5)
    window_eps = 0.10
    delta_t = 0.0
    jitter_J = 0.0

    guardrails = {
        "curve_lint": True,
        "no_skip": True,
        "pf_born_ties_only": True,
        "boolean_marks": True
    }

    # RNGs per seed for determinism
    rngs = {int(seed): np.random.Generator(np.random.PCG64(int(seed))) for seed in seeds}

    rows = []
    trials_total = int(K_per_seed) * len(seeds)

    for N in N_list:
        # Baseline vis0 at p=0
        unmarked_total_p0 = 0
        for seed in seeds:
            rng = rngs[int(seed)]
            marked = bernoulli_draws(0.0, int(K_per_seed), rng)  # always False
            unmarked_total_p0 += np.count_nonzero(~marked)
        vis0 = unmarked_total_p0 / trials_total

        for p in p_list:
            peff = 1.0 - (1.0 - float(p))**int(N)
            unmarked_total = 0
            neutral_total = 0
            for seed in seeds:
                rng = rngs[int(seed)]
                marked = bernoulli_draws(peff, int(K_per_seed), rng)
                unmarked = np.count_nonzero(~marked)
                unmarked_total += unmarked
                neutral_total += np.count_nonzero(marked)  # in the ties-only zone, marks → neutral
            vis = unmarked_total / trials_total
            vis_norm = vis / vis0 if vis0 > 0 else 0.0
            pred_binom = (1.0 - float(p))**int(N)
            pred_exp = math.exp(-float(p) * int(N))
            neutral_rate = neutral_total / trials_total
            rows.append(dict(p=float(p), N=int(N), trials=trials_total, vis=vis, vis0=vis0,
                             vis_norm=vis_norm, pred_binom=pred_binom, pred_exp=pred_exp,
                             neutral_rate=neutral_rate))

    grid_df = pd.DataFrame(rows).sort_values(["N","p"]).reset_index(drop=True)

    # Fits
    fitN_rows = []
    for N in N_list:
        sub = grid_df[grid_df.N == int(N)].copy()
        eps_floor = 1e-12
        y = -np.log(np.clip(sub["vis_norm"].values, eps_floor, 1.0))
        x = sub["p"].values
        slope, intercept, r2 = linfit(x, y)
        slope_rel_err = (slope - int(N)) / int(N) if N != 0 else 0.0
        fitN_rows.append(dict(N=int(N), slope=slope, intercept=intercept, R2=r2,
                              slope_rel_err=slope_rel_err,
                              pass_slope=bool(abs(slope_rel_err) <= 0.10),
                              pass_r2=bool(r2 >= 0.98)))
    fitN_df = pd.DataFrame(fitN_rows).sort_values("N").reset_index(drop=True)

    fitp_rows = []
    for p in p_list:
        if float(p) <= 0.0:  # exclude p=0
            continue
        sub = grid_df[grid_df.p == float(p)].copy()
        eps_floor = 1e-12
        y = -np.log(np.clip(sub["vis_norm"].values, eps_floor, 1.0))
        x = sub["N"].values.astype(float)
        slope, intercept, r2 = linfit(x, y)
        slope_rel_err = (slope - float(p)) / float(p) if float(p) != 0 else 0.0
        fitp_rows.append(dict(p=float(p), slope=slope, intercept=intercept, R2=r2,
                              slope_rel_err=slope_rel_err,
                              pass_slope=bool(abs(slope_rel_err) <= 0.10),
                              pass_r2=bool(r2 >= 0.98)))
    fitp_df = pd.DataFrame(fitp_rows).sort_values("p").reset_index(drop=True)

    rmse_binom = float(np.sqrt(np.mean((grid_df["vis_norm"] - grid_df["pred_binom"])**2)))
    rmse_exp = float(np.sqrt(np.mean((grid_df["vis_norm"] - grid_df["pred_exp"])**2)))
    max_abs_err = float(np.max(np.abs(grid_df["vis_norm"] - grid_df["pred_binom"])))

    pass_curve_match = rmse_binom < 0.02 and rmse_exp < 0.02
    pass_linearity = bool(np.all(fitN_df["pass_slope"])) and bool(np.all(fitN_df["pass_r2"]))                      and bool(np.all(fitp_df["pass_slope"])) and bool(np.all(fitp_df["pass_r2"]))
    pass_guardrails = all(guardrails.values())
    overall_pass = bool(pass_curve_match and pass_linearity and pass_guardrails)

    # Write artifacts
    grid_csv = outdir / "collisional_decoherence_grid.csv"
    fitN_csv = outdir / "fit_fixedN.csv"
    fitp_csv = outdir / "fit_fixedp.csv"
    summary_json = outdir / "collisional_summary.json"
    grid_df.to_csv(grid_csv, index=False)
    fitN_df.to_csv(fitN_csv, index=False)
    fitp_df.to_csv(fitp_csv, index=False)

    summary_payload = {
        "rmse_vs_binom": rmse_binom,
        "rmse_vs_exp": rmse_exp,
        "max_abs_err": max_abs_err,
        "fit_fixedN": fitN_df.to_dict(orient="records"),
        "fit_fixedp": fitp_df.to_dict(orient="records"),
        "guardrails": guardrails,
        "overall_pass": overall_pass
    }
    with open(summary_json, "w") as f:
        json.dump(summary_payload, f, indent=2)

    # Audit
    config = dict(
        p_list=list(map(float, p_list)), N_list=list(map(int, N_list)),
        K_per_seed=int(K_per_seed), seeds=list(map(int, seeds)),
        windows=dict(centers=list(windows_centers), eps=window_eps),
        delta_t=delta_t, jitter_J=jitter_J
    )
    config_bytes = json.dumps(config, sort_keys=True).encode("utf-8")
    config_sha256 = hashlib.sha256(config_bytes).hexdigest()

    audit_payload = {
        "guardrails": guardrails,
        "config_sha256": config_sha256,
        "notes": {
            "mark_generation": "Bernoulli with peff = 1 - (1-p)**N per trial (spec-allowed equivalence).",
            "ties_policy": "PF/Born used only when |C|=2 and marked=False; marked=True → neutral 50/50.",
            "cra_consistency": "Family labels tracked but not used in acceptance; inert as required."
        }
    }
    audits_dir = outdir.parent / "audits"
    audits_dir.mkdir(parents=True, exist_ok=True)
    with open(audits_dir / "audit.json", "w") as f:
        json.dump(audit_payload, f, indent=2)

    return {"grid_csv": str(grid_csv), "fit_fixedN": str(fitN_csv), "fit_fixedp": str(fitp_csv),
            "summary_json": str(summary_json), "audit_json": str(audits_dir / "audit.json")}
